/*
 * Decompiled with CFR 0.152.
 */
package com.codeborne.selenide.impl;

import com.codeborne.selenide.Browser;
import com.codeborne.selenide.Config;
import com.codeborne.selenide.DownloadsFolder;
import com.codeborne.selenide.Driver;
import com.codeborne.selenide.files.DownloadAction;
import com.codeborne.selenide.files.DownloadedFile;
import com.codeborne.selenide.files.FileFilter;
import com.codeborne.selenide.impl.Downloader;
import com.codeborne.selenide.impl.Downloads;
import com.codeborne.selenide.impl.FileHelper;
import com.codeborne.selenide.impl.WebElementSource;
import com.codeborne.selenide.impl.WindowsCloser;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.annotation.CheckReturnValue;
import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebElement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ParametersAreNonnullByDefault
public class DownloadFileToFolder {
    private static final Logger log = LoggerFactory.getLogger(DownloadFileToFolder.class);
    private static final String CHROME_TEMPORARY_FILE = "crdownload";
    private static final String FIREFOX_TEMPORARY_FILE = "part";
    private final Downloader downloader;
    private final WindowsCloser windowsCloser;

    DownloadFileToFolder(Downloader downloader, WindowsCloser windowsCloser) {
        this.downloader = downloader;
        this.windowsCloser = windowsCloser;
    }

    public DownloadFileToFolder() {
        this(new Downloader(), new WindowsCloser());
    }

    @CheckReturnValue
    @Nonnull
    public File download(WebElementSource anyClickableElement, WebElement clickable, long timeout, long incrementTimeout, FileFilter fileFilter, DownloadAction action) throws FileNotFoundException {
        WebDriver webDriver = anyClickableElement.driver().getWebDriver();
        long minimalIncrementTimeout = Math.max(incrementTimeout, 1000L);
        return this.windowsCloser.runAndCloseArisedWindows(webDriver, () -> this.clickAndWaitForNewFilesInDownloadsFolder(anyClickableElement, clickable, timeout, minimalIncrementTimeout, fileFilter, action));
    }

    @CheckReturnValue
    @Nonnull
    private File clickAndWaitForNewFilesInDownloadsFolder(WebElementSource anyClickableElement, WebElement clickable, long timeout, long incrementTimeout, FileFilter fileFilter, DownloadAction action) throws FileNotFoundException {
        Driver driver = anyClickableElement.driver();
        Config config = driver.config();
        long pollingInterval = Math.max(config.pollingInterval(), 50L);
        DownloadsFolder folder = driver.browserDownloadsFolder();
        if (folder == null) {
            throw new IllegalStateException("Downloads folder is not configured");
        }
        folder.cleanupBeforeDownload();
        long downloadStartedAt = System.currentTimeMillis();
        action.perform(driver, clickable);
        this.waitForNewFiles(fileFilter, folder, downloadStartedAt, timeout, incrementTimeout, pollingInterval);
        this.waitUntilDownloadsCompleted(driver.browser(), folder, fileFilter, timeout, incrementTimeout, pollingInterval);
        Downloads newDownloads = new Downloads(DownloadFileToFolder.newFiles(folder, downloadStartedAt));
        if (log.isInfoEnabled()) {
            log.info("Downloaded {}", (Object)newDownloads.filesAsString());
        }
        if (log.isDebugEnabled()) {
            log.debug("All downloaded files in {}: {}", (Object)folder, (Object)folder.files().stream().map(f -> f.getName()).collect(Collectors.joining("\n")));
        }
        File downloadedFile = newDownloads.firstDownloadedFile(timeout, fileFilter);
        return this.archiveFile(config, downloadedFile);
    }

    private void waitUntilDownloadsCompleted(Browser browser, DownloadsFolder folder, FileFilter filter, long timeout, long incrementTimeout, long pollingInterval) throws FileNotFoundException {
        if (browser.isChrome() || browser.isEdge()) {
            this.waitUntilFileDisappears(folder, CHROME_TEMPORARY_FILE, filter, timeout, incrementTimeout, pollingInterval);
        } else if (browser.isFirefox()) {
            this.waitUntilFileDisappears(folder, FIREFOX_TEMPORARY_FILE, filter, timeout, incrementTimeout, pollingInterval);
        } else {
            this.waitWhileFilesAreBeingModified(folder, timeout, pollingInterval);
        }
    }

    private void waitUntilFileDisappears(DownloadsFolder folder, String extension, FileFilter filter, long timeout, long incrementTimeout, long pollingInterval) throws FileNotFoundException {
        long start = System.currentTimeMillis();
        while (System.currentTimeMillis() - start <= timeout) {
            if (!folder.hasFiles(extension, filter)) {
                log.debug("No {} files found in {}, conclude download is completed", (Object)extension, (Object)folder);
                return;
            }
            log.debug("Found {} files in {}, waiting for {} ms...", new Object[]{extension, folder, pollingInterval});
            this.failFastIfNoChanges(folder, filter, start, timeout, incrementTimeout);
            this.pause(pollingInterval);
        }
        if (folder.hasFiles(extension, filter)) {
            String message = String.format("Folder %s still contains files %s after %s ms. Apparently, the downloading hasn't completed in time.", folder, extension, timeout);
            throw new FileNotFoundException(message);
        }
    }

    private void waitWhileFilesAreBeingModified(DownloadsFolder folder, long timeout, long pollingInterval) {
        Map<String, Long> times = folder.modificationTimes();
        long lastModifiedAt = System.currentTimeMillis();
        long start = System.currentTimeMillis();
        while (System.currentTimeMillis() - start < timeout) {
            Map<String, Long> newTimes = folder.modificationTimes();
            if (!newTimes.equals(times)) {
                log.debug("Files has been modified - old: {}, new: {}", times, newTimes);
                lastModifiedAt = System.currentTimeMillis();
                times = newTimes;
            } else {
                log.debug("Files has not been modified in last {} ms: {}", (Object)pollingInterval, times);
                if (System.currentTimeMillis() - lastModifiedAt > 1000L) {
                    log.debug("Files has not been modified during last {} ms.", (Object)(System.currentTimeMillis() - lastModifiedAt));
                    return;
                }
            }
            this.pause(pollingInterval);
        }
        log.warn("Files are still being modified during last {} ms.", (Object)(System.currentTimeMillis() - lastModifiedAt));
    }

    private void waitForNewFiles(FileFilter fileFilter, DownloadsFolder folder, long clickMoment, long timeout, long incrementTimeout, long pollingInterval) throws FileNotFoundException {
        Downloads downloads;
        long start = System.currentTimeMillis();
        while (System.currentTimeMillis() - start <= timeout && (downloads = new Downloads(DownloadFileToFolder.newFiles(folder, clickMoment))).files(fileFilter).isEmpty()) {
            this.failFastIfNoChanges(folder, fileFilter, start, timeout, incrementTimeout);
            this.pause(pollingInterval);
        }
    }

    private void failFastIfNoChanges(DownloadsFolder folder, FileFilter filter, long start, long timeout, long incrementTimeout) throws FileNotFoundException {
        long lastFileUpdate = folder.lastModificationTime().orElse(start);
        long filesHasNotBeenUpdatedForMs = System.currentTimeMillis() - lastFileUpdate;
        if (filesHasNotBeenUpdatedForMs > incrementTimeout) {
            String message = String.format("Failed to download file%s in %d ms: files in %s haven't been modified for %s ms.", filter.description(), timeout, folder, filesHasNotBeenUpdatedForMs);
            throw new FileNotFoundException(message);
        }
    }

    private void pause(long milliseconds) {
        try {
            Thread.sleep(milliseconds);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
    }

    @Nonnull
    private File archiveFile(Config config, File downloadedFile) {
        File uniqueFolder = this.downloader.prepareTargetFolder(config);
        File archivedFile = new File(uniqueFolder, downloadedFile.getName());
        FileHelper.moveFile(downloadedFile, archivedFile);
        return archivedFile;
    }

    private static List<DownloadedFile> newFiles(DownloadsFolder folder, long modifiedAfterTs) {
        return folder.files().stream().filter(File::isFile).filter(file -> DownloadFileToFolder.isFileModifiedLaterThan(file, modifiedAfterTs)).map(file -> new DownloadedFile((File)file, Collections.emptyMap())).collect(Collectors.toList());
    }

    static boolean isFileModifiedLaterThan(File file, long timestamp) {
        return file.lastModified() >= timestamp / 1000L * 1000L;
    }
}

