/*
 * Decompiled with CFR 0.152.
 */
package io.github.gitflowincrementalbuilder;

import io.github.gitflowincrementalbuilder.ChangedProjects;
import io.github.gitflowincrementalbuilder.DownstreamCalculator;
import io.github.gitflowincrementalbuilder.LazyValue;
import io.github.gitflowincrementalbuilder.config.Configuration;
import io.github.gitflowincrementalbuilder.jgit.GitProvider;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.project.MavenProject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
@Named
class UnchangedProjectsRemover {
    private static final String MAVEN_TEST_SKIP = "maven.test.skip";
    private static final String MAVEN_TEST_SKIP_EXEC = "skipTests";
    private static final String GOAL_TEST_JAR = "test-jar";
    private Logger logger = LoggerFactory.getLogger(UnchangedProjectsRemover.class);
    @Inject
    private ChangedProjects changedProjects;
    @Inject
    private DownstreamCalculator downstreamCalculator;
    @Inject
    private GitProvider gitProvider;

    UnchangedProjectsRemover() {
    }

    public void act(Configuration config) {
        try {
            this.doAct(config);
        }
        finally {
            this.downstreamCalculator.clearCache();
        }
    }

    private void doAct(Configuration config) {
        Set<MavenProject> selected;
        LazyMavenProjectComparator projectComparator = new LazyMavenProjectComparator(config.mavenSession);
        config.logImpactedTo.ifPresent(logFilePath -> {
            try {
                Files.deleteIfExists(logFilePath);
            }
            catch (IOException e) {
                this.logger.warn("Could not delete '" + logFilePath + "', file might contain outdated projects!", (Throwable)e);
            }
        });
        if (config.disableSelectedProjectsHandling) {
            selected = Collections.emptySet();
        } else {
            selected = ProjectSelectionUtil.gatherSelectedProjects(config.mavenSession);
            if (this.onlySelectedModulesPresent(selected, config.mavenSession)) {
                this.printDelimiter();
                this.logger.info("Building explicitly selected projects (without any adjustment): {}", (Object)config.mavenSession.getProjects().stream().map(MavenProject::getArtifactId).collect(Collectors.joining(", ")));
                config.logImpactedTo.ifPresent(logFilePath -> this.writeImpactedLogFile(selected, (Path)logFilePath, projectComparator, config));
                return;
            }
            if (!config.mavenSession.getRequest().isRecursive() || this.onlySingleLeafModulePresent(config)) {
                this.printDelimiter();
                this.logger.info("Building single project (without any adjustment): {}", (Object)config.currentProject.getArtifactId());
                config.logImpactedTo.ifPresent(logFilePath -> this.writeImpactedLogFile(Set.of(config.currentProject), (Path)logFilePath, projectComparator, config));
                return;
            }
        }
        Set<MavenProject> changed = this.changedProjects.get(config);
        this.printDelimiter();
        if (changed.isEmpty()) {
            this.handleNoChangesDetected(selected, projectComparator, config);
            config.logImpactedTo.ifPresent(logFilePath -> this.writeImpactedLogFile(Collections.emptySet(), (Path)logFilePath, projectComparator, config));
            return;
        }
        Set<MavenProject> impacted = this.calculateImpactedProjects(selected, changed, config);
        LazyValue<List> lazyDownstreamProjects = new LazyValue<List>(() -> impacted.stream().filter(Predicate.not(changed::contains)).collect(Collectors.toList()));
        if (!config.argsForDownstreamModules.isEmpty()) {
            lazyDownstreamProjects.get().forEach(proj -> config.argsForDownstreamModules.forEach(proj.getProperties()::setProperty));
        }
        config.logImpactedTo.ifPresent(logFilePath -> this.writeImpactedLogFile(impacted, (Path)logFilePath, projectComparator, config));
        LazyValue<List> lazyUpstreamProjects = new LazyValue<List>(() -> config.mavenSession.getProjects().stream().filter(Predicate.not(impacted::contains)).collect(Collectors.toList()));
        if (!config.buildAll) {
            this.modifyProjectList(selected, changed, impacted, projectComparator, config);
        } else {
            lazyUpstreamProjects.get().forEach(proj -> this.applyUpstreamModuleArgs((MavenProject)proj, config));
        }
        if (config.logProjectsMode != Configuration.LogProjectsMode.NONE) {
            this.logProjects(changed, "Changed", projectComparator, config.mavenSession);
        }
        if (config.logProjectsMode == Configuration.LogProjectsMode.IMPACTED || config.logProjectsMode == Configuration.LogProjectsMode.ALL) {
            this.logProjects(lazyDownstreamProjects.get(), "Downstream", projectComparator, config.mavenSession);
        }
        if (config.logProjectsMode == Configuration.LogProjectsMode.ALL) {
            this.logProjects(lazyUpstreamProjects.get(), "Upstream", projectComparator, config.mavenSession);
        }
    }

    private boolean onlySelectedModulesPresent(Set<MavenProject> selected, MavenSession mavenSession) {
        return !selected.isEmpty() && mavenSession.getProjects().equals(new ArrayList<MavenProject>(selected));
    }

    private boolean onlySingleLeafModulePresent(Configuration config) {
        return config.mavenSession.getProjects().size() == 1 && config.currentProject.getModel().getModules().isEmpty();
    }

    private void handleNoChangesDetected(Set<MavenProject> selected, LazyMavenProjectComparator projectComparator, Configuration config) {
        if (!selected.isEmpty()) {
            this.logger.info("No changed artifacts detected: Building just explicitly selected projects (and their upstream and/or downstream, if requested).");
            Set selectedAndDownstream = selected.stream().flatMap(proj -> this.downstreamCalculator.streamProjectWithDownstreamProjects((MavenProject)proj, config)).collect(Collectors.toSet());
            if (Configuration.isMakeBehaviourActive("make-upstream", config.mavenSession)) {
                if (config.buildUpstreamMode == Configuration.BuildUpstreamMode.NONE) {
                    config.mavenSession.setProjects(selectedAndDownstream.stream().sorted(projectComparator).collect(Collectors.toList()));
                } else {
                    config.mavenSession.getProjects().stream().filter(Predicate.not(selectedAndDownstream::contains)).forEach(proj -> this.applyUpstreamModuleArgs((MavenProject)proj, config));
                }
            }
            if (Configuration.isMakeBehaviourActive("make-downstream", config.mavenSession) && !config.buildDownstream) {
                config.mavenSession.setProjects(config.mavenSession.getProjects().stream().filter(proj -> selected.contains(proj) || !selectedAndDownstream.contains(proj)).collect(Collectors.toList()));
            }
        } else if (config.buildAllIfNoChanges) {
            this.logger.info("No changed artifacts detected: Building all modules in buildAll mode.");
            this.logger.info("- skip tests: {}", (Object)config.skipTestsForUpstreamModules);
            this.logger.info("- additional args: {}", config.argsForUpstreamModules);
            config.mavenSession.getProjects().stream().forEach(proj -> this.applyUpstreamModuleArgs((MavenProject)proj, config));
        } else {
            this.logger.info("No changed artifacts detected: Executing validate goal on current project only, skipping all submodules.");
            config.mavenSession.setProjects(Collections.singletonList(this.applyUpstreamModuleArgs(config.currentProject, config)));
            config.mavenSession.getGoals().clear();
            config.mavenSession.getGoals().add("validate");
        }
    }

    private Set<MavenProject> calculateImpactedProjects(Set<MavenProject> selected, Set<MavenProject> changed, Configuration config) {
        Stream<Object> impacted;
        Stream stream = impacted = selected.isEmpty() ? changed.stream() : selected.stream();
        if (config.buildAll || config.buildDownstream) {
            impacted = impacted.flatMap(proj -> this.downstreamCalculator.streamProjectWithDownstreamProjects((MavenProject)proj, config));
        }
        return impacted.filter(config.mavenSession.getProjects()::contains).collect(Collectors.toCollection(LinkedHashSet::new));
    }

    private void writeImpactedLogFile(Set<MavenProject> impacted, Path logFilePath, LazyMavenProjectComparator projectComparator, Configuration config) {
        List projectsToLog;
        if (impacted.isEmpty()) {
            projectsToLog = Collections.emptyList();
        } else {
            Path projectRootDir = this.gitProvider.getProjectRoot(config);
            projectsToLog = impacted.stream().sorted(projectComparator).map(proj -> projectRootDir.relativize(proj.getBasedir().toPath()).toString()).collect(Collectors.toList());
        }
        this.logger.debug("Writing impacted projects to {}: {}", (Object)logFilePath, projectsToLog);
        try {
            Path parentDir = logFilePath.toAbsolutePath().getParent();
            if (parentDir != null && !Files.exists(parentDir, new LinkOption[0])) {
                Files.createDirectories(parentDir, new FileAttribute[0]);
            }
            Files.write(logFilePath, projectsToLog, StandardCharsets.UTF_8, StandardOpenOption.CREATE);
        }
        catch (IOException e) {
            throw new IllegalStateException("Failed to write impacted projects to " + logFilePath + ": " + impacted, e);
        }
    }

    private void modifyProjectList(Set<MavenProject> selected, Set<MavenProject> changed, Set<MavenProject> impacted, LazyMavenProjectComparator projectComparator, Configuration config) {
        Set<MavenProject> rebuild = this.calculateRebuildProjects(selected, changed, impacted, config);
        if (rebuild.isEmpty()) {
            this.handleNoChangesDetected(selected, projectComparator, config);
        } else {
            if (!config.forceBuildModules.isEmpty() || !config.forceBuildModulesConditionally.isEmpty()) {
                List conditionalPatterns = config.forceBuildModulesConditionally.entrySet().stream().filter(entry -> impacted.stream().anyMatch(proj -> ((Pattern)entry.getKey()).matcher(proj.getArtifactId()).matches())).map(Map.Entry::getValue).collect(Collectors.toList());
                Set forceBuildModules = config.mavenSession.getProjects().stream().filter(Predicate.not(rebuild::contains)).filter(proj -> this.matchesAny(proj.getArtifactId(), config.forceBuildModules) || this.matchesAny(proj.getArtifactId(), conditionalPatterns)).map(proj -> this.applyUpstreamModuleArgs((MavenProject)proj, config)).collect(Collectors.toCollection(LinkedHashSet::new));
                rebuild.addAll(forceBuildModules);
            }
            config.mavenSession.setProjects(rebuild.stream().sorted(projectComparator).collect(Collectors.toList()));
        }
    }

    private Set<MavenProject> calculateRebuildProjects(Set<MavenProject> selected, Set<MavenProject> changed, Set<MavenProject> impacted, Configuration config) {
        Set<MavenProject> upstreamRequiringProjects;
        Configuration.BuildUpstreamMode buildUpstreamMode = config.buildUpstreamMode;
        switch (buildUpstreamMode) {
            case NONE: {
                return impacted;
            }
            case CHANGED: {
                upstreamRequiringProjects = selected.isEmpty() ? changed : selected;
                break;
            }
            case IMPACTED: {
                upstreamRequiringProjects = impacted;
                break;
            }
            default: {
                throw new IllegalStateException("Unsupported BuildUpstreamMode: " + buildUpstreamMode);
            }
        }
        Set upstreamProjects = upstreamRequiringProjects.stream().flatMap(proj -> this.streamUpstreamProjects((MavenProject)proj, config.mavenSession)).filter(Predicate.not(impacted::contains)).peek(proj -> this.applyUpstreamModuleArgs((MavenProject)proj, config)).collect(Collectors.toCollection(LinkedHashSet::new));
        return config.mavenSession.getProjects().stream().filter(proj -> impacted.contains(proj) || upstreamProjects.contains(proj)).collect(Collectors.toCollection(LinkedHashSet::new));
    }

    private MavenProject applyUpstreamModuleArgs(MavenProject mavenProject, Configuration config) {
        Properties projectProperties = mavenProject.getProperties();
        if (config.skipTestsForUpstreamModules) {
            if (this.projectDeclaresTestJarGoal(mavenProject)) {
                this.logger.debug("Will not skip test compilation of module {} because it has a {} goal.", (Object)mavenProject.getArtifactId(), (Object)GOAL_TEST_JAR);
                projectProperties.setProperty(MAVEN_TEST_SKIP_EXEC, Boolean.TRUE.toString());
            } else {
                projectProperties.setProperty(MAVEN_TEST_SKIP, Boolean.TRUE.toString());
            }
        }
        config.argsForUpstreamModules.forEach(projectProperties::setProperty);
        return mavenProject;
    }

    private boolean projectDeclaresTestJarGoal(MavenProject project) {
        return project.getBuildPlugins().stream().flatMap(p -> p.getExecutions().stream()).flatMap(e -> e.getGoals().stream()).anyMatch(GOAL_TEST_JAR::equals);
    }

    private void logProjects(Collection<MavenProject> projects, String titlePrefix, LazyMavenProjectComparator projectComparator, MavenSession mavenSession) {
        if (projects.isEmpty()) {
            return;
        }
        this.logger.info("{} artifactIds ({}):", (Object)titlePrefix, (Object)projects.size());
        this.logger.info("");
        projects.stream().sorted(projectComparator).map(proj -> {
            Object entry = proj.getArtifactId();
            if (!mavenSession.getProjects().contains(proj)) {
                entry = (String)entry + " (but deselected)";
            }
            return "- " + (String)entry;
        }).forEach(arg_0 -> ((Logger)this.logger).info(arg_0));
        this.logger.info("");
    }

    private void printDelimiter() {
        this.logger.info("------------------------------------------------------------------------");
    }

    private Stream<MavenProject> streamUpstreamProjects(MavenProject project, MavenSession mavenSession) {
        return mavenSession.getProjectDependencyGraph().getUpstreamProjects(project, true).stream();
    }

    private boolean matchesAny(String str, Collection<Pattern> patterns) {
        return !patterns.isEmpty() && patterns.stream().anyMatch(pattern -> pattern.matcher(str).matches());
    }

    static class LazyMavenProjectComparator
    implements Comparator<MavenProject> {
        private final MavenSession mavenSession;
        private Map<MavenProject, Integer> indexMap;

        public LazyMavenProjectComparator(MavenSession mavenSession) {
            this.mavenSession = mavenSession;
        }

        @Override
        public int compare(MavenProject proj1, MavenProject proj2) {
            if (this.indexMap == null) {
                List projects = this.mavenSession.getProjects();
                this.indexMap = IntStream.range(0, projects.size()).boxed().collect(Collectors.toMap(projects::get, i -> i));
                List allProjects = this.mavenSession.getAllProjects();
                if (allProjects.size() > projects.size()) {
                    for (int i2 = 0; i2 < allProjects.size(); ++i2) {
                        MavenProject proj = (MavenProject)allProjects.get(i2);
                        if (this.indexMap.containsKey(proj)) continue;
                        this.indexMap.put(proj, i2 + 100000);
                    }
                }
            }
            return this.indexMap.getOrDefault(proj1, Integer.MAX_VALUE).compareTo(this.indexMap.getOrDefault(proj2, Integer.MAX_VALUE));
        }
    }

    private static class ProjectSelectionUtil {
        private ProjectSelectionUtil() {
        }

        static Set<MavenProject> gatherSelectedProjects(MavenSession mavenSession) {
            List selectors = mavenSession.getRequest().getSelectedProjects();
            if (selectors.isEmpty()) {
                return Collections.emptySet();
            }
            File reactorDirectory = Optional.ofNullable(mavenSession.getRequest().getBaseDirectory()).map(File::new).orElse(null);
            return mavenSession.getProjects().stream().filter(proj -> selectors.stream().anyMatch(sel -> ProjectSelectionUtil.matchesSelector(proj, sel, reactorDirectory))).collect(Collectors.toCollection(LinkedHashSet::new));
        }

        private static boolean matchesSelector(MavenProject project, String selector, File reactorDirectory) {
            if (selector.contains(":")) {
                String id = ":" + project.getArtifactId();
                if (id.equals(selector)) {
                    return true;
                }
                id = project.getGroupId() + id;
                if (id.equals(selector)) {
                    return true;
                }
            } else if (reactorDirectory != null) {
                File selectedProject = new File(new File(reactorDirectory, selector).toURI().normalize());
                if (selectedProject.isFile()) {
                    return selectedProject.equals(project.getFile());
                }
                if (selectedProject.isDirectory()) {
                    return selectedProject.equals(project.getBasedir());
                }
            }
            return false;
        }
    }
}

